#' @title  Variational Bayesian Algorithm for Multi-Source Heterogeneous Models.
#' @description
#'  This package implements a variational Bayesian algorithm for high-dimensional multi-source
#'    heterogeneous linear models. It simultaneously performs parameter estimation and variable selection.
#'    The algorithm supports two model settings: (1) local models, where variable selection is only      #'    applied to homogeneous coefficients, and (2) global models, where variable selection is also       #'    performed on heterogeneous coefficients. Two forms of Spike-and-Slab priors are available: the     #'    Laplace distribution and the Gaussian distribution as the Slab component.
#'
#'
#' @param X Homogeneous covariates
#' @param Z Heterogeneous covariates
#' @param Y Response covariates
#' @param global Indicates whether variable selection is required for het coefficients,
#'               if TRUE, Variable selection will be made for het coefficients.
#'
#' @param prior Forms of Slab distribution in Spike-and-Slab prior, "laplace" or "gauss".
#' @param max_iter Maximum number of iterations, Defaut:1000
#' @param tol Algorithm convergence tolerance, Defaut:1e-6
#' @param a A prior of Beta distribution, Defaut:1
#' @param b A prior of Beta distribution, Defaut:10
#' @param lambda A prior of Laplace distribution, Defaut:1
#'
#' @return \item{mu_hom}{The mean of the homogeneous coefficients}
#'         \item{sigma_hom}{The variance of homogeneous coefficients}
#'         \item{gamma_hom}{Selection indicators for homogeneous coefficients}
#'         \item{mu_het}{The mean of the heterogeneous coefficients}
#'         \item{sigma_het}{The variance of heterogeneous coefficients}
#'         \item{gamma_het}{Selection indicators for heterogeneous coefficients (NULL for local models)}
#' @examples
#' # Simulate multi-source heterogeneous data
#' n <- 50  # number of samples per source
#' K <- 3   # number of sources
#' p <- 100  # number of homogeneous covariates
#' q <- 5    # number of heterogeneous covariates
#'
#' set.seed(1)
#' theta <- matrix(c(c(-1,0.5,1,-0.5,2),rep(0,p-5)), ncol = 1)
#' beta <- matrix(1, nrow = q, ncol = K)
#' for (k in 1:K) {
#'   beta[,k] <- matrix(c(rep(log(k+1),5),rep(0,q-5)), ncol = 1)
#' }
#'
#' zdata <- MASS::mvrnorm(K*n, rep(0,q), diag(q))
#' Z <- array(data=zdata,dim=c(n,q,K))
#' xdata <- MASS::mvrnorm(K*n, rep(0,p), diag(p))
#' X <- array(data=xdata,dim=c(n,p,K))
#' Y <- matrix(0, nrow = n, ncol = K)
#'
#' for (k in 1:K) {
#'   Y[,k] <- MASS::mvrnorm(1, X[,,k]%*%theta+Z[,,k]%*%beta[,k], diag(n))
#' }
#'
#' # Fit local model with Laplace prior
#' res <- vbms(X, Z, Y, global=FALSE, prior='laplace')
#'
#' # View results
#' print(head(res$mu_hom))      # Homogeneous coefficients mean
#' print(head(res$gamma_hom))   # Homogeneous variable selection
#' print(res$mu_het)            # Heterogeneous coefficients mean
#'
#' @importFrom MASS mvrnorm
#' @importFrom stats optim rnorm
#' @export

vbms <- function(X, Z, Y, global,prior,max_iter=1000, tol=1e-6, a=1, b=10, lambda=1){
  if (global==FALSE){
    if (prior=='laplace'){
      res <- vb_lap_local(X, Z, Y, max_iter, tol, a, b, lambda)
    }else{
      res <- vb_gauss_local(X, Z, Y, max_iter, tol, a, b, lambda)
    }
    mu_hom <- res$mu
    sigma_hom <- res$sigma
    gamma_hom <- res$gamma
    mu_het <- res$m
    sigma_het <- res$s2
    gamma_het <- NULL
  }else{
    if (prior=='laplace'){
      res <- vb_lap_global(X, Z, Y, max_iter, tol, a, b, lambda)
    }else{
      res <- vb_gauss_global(X, Z, Y, max_iter, tol, a, b, lambda)
    }
    mu_hom <- res$mu1
    sigma_hom <- res$sigma1
    gamma_hom <- res$gamma1
    mu_het <- res$mu2
    sigma_het <- res$sigma2
    gamma_het <- res$gamma2
  }
  result <- list(mu_hom=mu_hom,sigma_hom=sigma_hom,gamma_hom=gamma_hom,
                 mu_het=mu_het,sigma_het=sigma_het,gamma_het=gamma_het)
  return(result)
}
