## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----setup--------------------------------------------------------------------
library(twoCoprimary)

## ----two_continuous_example---------------------------------------------------
# Example: Two continuous endpoints with correlation rho = 0.5
ss2Continuous(
  delta1 = 0.5,  # Standardized effect size for endpoint 1
  delta2 = 0.5,  # Standardized effect size for endpoint 2
  sd1 = 1,       # Standard deviation for endpoint 1
  sd2 = 1,       # Standard deviation for endpoint 2
  rho = 0.5,     # Correlation between endpoints
  r = 1,         # Balanced allocation
  alpha = 0.025,
  beta = 0.2,
  known_var = TRUE
)

## ----two_binary_approx_example------------------------------------------------
# Example: Two binary endpoints
ss2BinaryApprox(
  p11 = 0.7, p12 = 0.6,  # Endpoint 1 and 2 in treatment group
  p21 = 0.4, p22 = 0.3,  # Endpoint 1 and 2 in control group
  rho1 = 0.5,            # Correlation in treatment group
  rho2 = 0.5,            # Correlation in control group
  r = 1,                 # Balanced allocation
  alpha = 0.025,
  beta = 0.2,
  Test = "AN"
)

## ----two_binary_exact_example-------------------------------------------------
# Example: Exact methods for small samples
ss2BinaryExact(
  p11 = 0.7, p12 = 0.6,
  p21 = 0.4, p22 = 0.3,
  rho1 = 0.5, rho2 = 0.5,
  r = 1,
  alpha = 0.025,
  beta = 0.2,
  Test = "Fisher"  # or "Chisq", "Fisher-midP", "Z-pool", "Boschloo"
)

## ----mixed_cont_binary_example------------------------------------------------
# Example: Continuous + Binary endpoints
ss2MixedContinuousBinary(
  delta = 0.5,           # Effect size for continuous endpoint
  sd = 1,                # Standard deviation
  p1 = 0.7,              # Success probability in treatment group
  p2 = 0.4,              # Success probability in control group
  rho = 0.5,             # Biserial correlation
  r = 1,
  alpha = 0.025,
  beta = 0.2,
  Test = "AN"
)

## ----mixed_count_cont_example-------------------------------------------------
# Example: Count (exacerbations) + Continuous (FEV1)
ss2MixedCountContinuous(
  r1 = 1.0, r2 = 1.25,   # Count rates (events per unit time)
  nu = 0.8,              # Dispersion parameter
  t = 1,                 # Follow-up time
  mu1 = -50, mu2 = 0,    # Continuous means
  sd = 250,              # Standard deviation
  rho1 = 0.5, rho2 = 0.5, # Correlations
  r = 1,
  alpha = 0.025,
  beta = 0.2
)

## ----correlation_impact-------------------------------------------------------
# Sample size at different correlation levels
correlations <- c(0, 0.3, 0.5, 0.8)
results <- sapply(correlations, function(rho) {
  ss2Continuous(
    delta1 = 0.5, delta2 = 0.5,
    sd1 = 1, sd2 = 1,
    rho = rho, r = 1,
    alpha = 0.025, beta = 0.2,
    known_var = TRUE
  )$N
})

data.frame(
  Correlation = correlations,
  Total_N = results,
  Reduction = paste0(round((1 - results/results[1]) * 100, 1), "%")
)

