## ---- include = FALSE---------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----setup--------------------------------------------------------------------
library(telemetR)

## ---- load some data from the package, echo=TRUE------------------------------
fish <- telemetR::fish
receivers <- telemetR::receivers
detects <- telemetR::filtered_detections

## ---- format the data to make it easier to use the package, echo = TRUE-------
local_tz <- "America/Los_Angeles"

fish <- format_org(fish, 
                   var_Id = "TagCode", 
                   var_release = "Release_Date",
                   var_tag_life = "TagLife",
                   var_ping_rate = "PRI",
                   local_time_zone = local_tz,
                   time_format = "%Y-%m-%d %H:%M:%S")
receivers <- format_receivers(receivers,
                              var_receiver_serial = "receiver_serial_number",
                              var_receiver_make = "receiver_make",
                              var_receiver_deploy = "receiver_start",
                              var_receiver_retrieve = "receiver_end",
                              local_time_zone = local_tz,
                              time_format = "%m/%d/%Y %H:%M:%S")
detects <- format_detects(detects,
                          var_Id = "Tag_Code",
                          var_datetime_local = "DateTime_Local",
                          var_receiver_serial = "ReceiverSN",
                          local_time_zone = local_tz,
                          time_format = "%Y-%m-%d %H:%M:%S")

## ---- Begin the blanking period analysis, echo = TRUE-------------------------
# Build a new df which uses a range of blanking times to reprocess the original
# detection data (this is a slow process, especially for large datasets or
# large sequences of n)

blanked <- blanking_event(detects,
                     #Each general location is distinct enough to be a site
                     var_site = "receiver_general_location",
                     #Group the data by species, run, and origin (optional)
                     var_groups = "fish_type",
                     var_Id = "Tag_Code",
                     var_datetime = "DateTime_Local",
                     var_ping_rate = "tag_pulse_rate_interval_nominal",
                     n_val = seq(1,1000,5),
                     #Pulse rates for these tags are in seconds
                     time_unit = "secs"
                     )
head(blanked)

## ---- Compare the results of the blanking analysis to a set of times, echo = TRUE----

# This function compares the durations of the events created above. The function
# collects all the events for each given blanking period and compares each
# event duration to a sequence of given times. It then calculates the proportion
# of events which are longer than each given time for each blanking period for
# each organism group

compared <- duration_compare(blanked, 
                             var_groups = "fish_type", 
                             time_seq = seq(1,10000,10))

head(compared)

## ----plot the proportion of events with durations greater than t, echo = TRUE----
residence_plot(compared, var_groups = "fish_type", time_unit = "secs")


## ---- echo = TRUE-------------------------------------------------------------
# NOTE: The curves last longer than the longest time! This should indicate we
# should add more times to test, and may need shorter intervals of n.

## ----next calculate the renormalized sum of squares, echo = TRUE--------------

# Renormalized Sum of Squares
rSSR <- renorm_SSR(compared, var_groups = "fish_type")

head(rSSR)


## ----next calculate the convergence thresholds, echo = TRUE-------------------

# Thresholds for 99% and 99.5% of total range
thresh <- telemetR::conv_thresholds(rSSR, var_groups = "fish_type", 
                                    thresh_levels = c(0.01,0.005))


## ----finally determine optimum mbps and plot rSSR results, echo = TRUE--------
optimums <- opt_mbp(rSSR, thresh)

rSSR_plot(rSSR, optimums, var_groups = "fish_type")
# Note: This would be a bad choice of maximum blanking period. We have not yet 
# reached convergence

